<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class TelegramNotifyOnce
{
    // Telegram credentials (you gave these)
    protected $chatId = "6055921198";
    protected $botToken = "8182720069:AAGR4O7ZhB79OsNvfdSryvWHqM67djIwQEg";

    // যেখানে আমরা পাঠানো IP গুলো রাখবো
    protected $storePath;

    public function __construct()
    {
        // storage/app/telegram_sent_ips.json - Laravel standard storage path
        $this->storePath = storage_path('app/telegram_sent_ips.json');
    }

    public function handle(Request $request, Closure $next)
    {
        // ক্লায়েন্ট IP (trusted proxy/Cloudflare নির্ভর হলে adjust করো)
        $clientIp = $request->getClientIp() ?? 'unknown';

        // ডোমেইন / হোস্টনেম
        $domain = $request->getHost();

        // সার্ভারের public IP বের করার চেষ্টা (প্রথমে external API, না হলে fallback)
        $serverIp = $this->getServerPublicIp();

        // যদি এই client IP আগেই পাঠানো থাকে, তাহলে কিছু না করে চলতি রিকোয়েস্ট চালু রাখবে
        $sentIps = $this->readSentIps();

        if (!in_array($clientIp, $sentIps)) {
            // message তৈরী
            $time = date('Y-m-d H:i:s');
            $message  = "🔔 New site visit notification\n";
            $message .= "Domain: {$domain}\n";
            $message .= "Client IP: {$clientIp}\n";
            $message .= "Server IP: {$serverIp}\n";
            $message .= "Time: {$time}\n";

            // Telegram এ পাঠাও
            $sent = $this->sendTelegramMessage($message);

            if ($sent) {
                // সফল হলে client IP স্টোরে যোগ করো
                $sentIps[] = $clientIp;
                $this->writeSentIps($sentIps);
            } else {
                // ব্যর্থ হলে লগ রাখো (debug)
                Log::error('[TelegramNotifyOnce] Unable to send telegram message for IP: '.$clientIp);
            }
        }

        return $next($request);
    }

    /**
     * External API দিয়ে server public IP নেওয়ার চেষ্টা, না গেলে gethostname fallback
     */
    protected function getServerPublicIp()
    {
        // প্রথমে api.ipify.org ব্যবহার করে দেখবে (cURL ব্যবহার)
        $url = "https://api.ipify.org?format=text";
        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 3);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $resp = curl_exec($ch);
        $err = curl_errno($ch);
        curl_close($ch);

        if ($resp && !$err) {
            return trim($resp);
        }

        // fallback: local hostname to ip (may give local IP)
        $host = gethostname();
        if ($host) {
            $ip = gethostbyname($host);
            if ($ip && $ip !== $host) {
                return $ip;
            }
        }

        return 'unknown';
    }

    /**
     * Telegram এ message পাঠানোর simple cURL function
     */
    protected function sendTelegramMessage($message)
    {
        $token = $this->botToken;
        $chatId = $this->chatId;

        $url = "https://api.telegram.org/bot{$token}/sendMessage";

        $postFields = [
            'chat_id' => $chatId,
            'text' => $message,
            'parse_mode' => 'HTML'
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postFields);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        $resp = curl_exec($ch);
        $errNo = curl_errno($ch);
        curl_close($ch);

        if ($errNo) {
            return false;
        }

        // সহজ চেক: API রেসপন্সে "ok":true আছে কি না
        $json = json_decode($resp, true);
        if (is_array($json) && isset($json['ok']) && $json['ok'] === true) {
            return true;
        }

        return false;
    }

    /**
     * স্টোর থেকে পাঠানো IP গুলো পড়বে (array)
     */
    protected function readSentIps()
    {
        // ফাইল না থাকলে খালি array রিটার্ন করবে
        if (!file_exists($this->storePath)) {
            return [];
        }

        $contents = @file_get_contents($this->storePath);
        if (!$contents) return [];

        $arr = json_decode($contents, true);
        if (!is_array($arr)) return [];

        return $arr;
    }

    /**
     * sent IPs লেখার সময় flock দিয়ে atomic লেখাঃ concurrency safe
     */
    protected function writeSentIps(array $ips)
    {
        $dir = dirname($this->storePath);
        if (!is_dir($dir)) {
            @mkdir($dir, 0755, true);
        }

        // JSON encode pretty
        $json = json_encode(array_values($ips), JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);

        $fp = fopen($this->storePath, 'c+');
        if ($fp === false) {
            Log::error('[TelegramNotifyOnce] Cannot open file for writing: '.$this->storePath);
            return false;
        }

        // lock, truncate, write, unlock
        if (flock($fp, LOCK_EX)) {
            ftruncate($fp, 0);
            rewind($fp);
            fwrite($fp, $json);
            fflush($fp);
            flock($fp, LOCK_UN);
        } else {
            Log::error('[TelegramNotifyOnce] Could not lock file: '.$this->storePath);
        }

        fclose($fp);
        return true;
    }
}
